/**
 * Cookie-Scanner für App bis Web - Super Simpel DSGVO
 * Protokolliert Cookie-Änderungen im Frontend für Admin-Debug
 */
; (function () {
  'use strict'

  // Prüfen ob Scanner-Config vorhanden
  if (typeof abwSsdDebug === 'undefined' || !abwSsdDebug.enabled) {
    return
  }

  // Bekannte Cookies mit Beschreibungen
  // Basiert auf: https://developer.wordpress.org/advanced-administration/wordpress/cookies/
  var knownCookies = {
    // Dieses Plugin
    'appbisweb-cookieconsent': { safe: true, cat: 'Plugin', desc: 'Cookie-Einwilligung (Super Simpel DSGVO)' },

    // WordPress Core
    'wordpress_': { safe: true, cat: 'WordPress', desc: 'Admin-Authentifizierung', prefix: true },
    'wordpress_logged_in_': { safe: true, cat: 'WordPress', desc: 'Login-Status', prefix: true },
    'wordpress_sec_': { safe: true, cat: 'WordPress', desc: 'Admin-Sicherheit', prefix: true },
    'wp-settings-': { safe: true, cat: 'WordPress', desc: 'Admin-Einstellungen', prefix: true },
    'wp_lang': { safe: true, cat: 'WordPress', desc: 'Spracheinstellung' },
    'wordpress_test_cookie': { safe: true, cat: 'WordPress', desc: 'Cookie-Test' },
    'comment_author_': { safe: true, cat: 'WordPress', desc: 'Kommentar-Name', prefix: true },

    // WooCommerce
    'woocommerce_': { safe: true, cat: 'WooCommerce', desc: 'Shop-Funktionalität', prefix: true },
    'wp_woocommerce_session_': { safe: true, cat: 'WooCommerce', desc: 'Warenkorb-Session', prefix: true },
    'store_notice': { safe: true, cat: 'WooCommerce', desc: 'Shop-Hinweis' },
    'wc_': { safe: true, cat: 'WooCommerce', desc: 'WooCommerce', prefix: true },

    // WPML / Polylang
    'wp-wpml_current_language': { safe: true, cat: 'WPML', desc: 'Sprachauswahl' },
    '_icl_': { safe: true, cat: 'WPML', desc: 'Spracheinstellung', prefix: true },
    'pll_language': { safe: true, cat: 'Polylang', desc: 'Sprachauswahl' },

    // Sicherheit
    'wfvt_': { safe: true, cat: 'Wordfence', desc: 'Sicherheits-Token', prefix: true },
    'wordfence': { safe: true, cat: 'Wordfence', desc: 'Sicherheit', prefix: true },
    '__cf_bm': { safe: true, cat: 'Cloudflare', desc: 'Bot-Schutz' },
    '__cfruid': { safe: true, cat: 'Cloudflare', desc: 'Rate-Limiting' },
    'cf_clearance': { safe: true, cat: 'Cloudflare', desc: 'Challenge bestanden' },

    // Cookie-Plugins
    'cookielawinfo': { safe: true, cat: 'Cookie-Plugin', desc: 'Einwilligung', prefix: true },
    'viewed_cookie_policy': { safe: true, cat: 'Cookie-Plugin', desc: 'Banner gesehen' },
    'cmplz_': { safe: true, cat: 'Complianz', desc: 'Einwilligung', prefix: true },
    'moove_gdpr_popup': { safe: true, cat: 'Cookie-Plugin', desc: 'Einwilligung' },

    // Zahlungsanbieter (notwendig)
    '__stripe_': { safe: true, cat: 'Stripe', desc: 'Zahlungs-Sicherheit', prefix: true },
    'PYPF': { safe: true, cat: 'PayPal', desc: 'Zahlung' },

    // Google Analytics
    '_ga': { safe: false, cat: 'Google Analytics', desc: 'Besucher-ID (2 Jahre)', tracking: true },
    '_ga_': { safe: false, cat: 'Google Analytics', desc: 'GA4 Session', prefix: true, tracking: true },
    '_gid': { safe: false, cat: 'Google Analytics', desc: 'Besucher-ID (24h)', tracking: true },
    '_gat': { safe: false, cat: 'Google Analytics', desc: 'Anfrage-Limit', tracking: true },
    '_gac_': { safe: false, cat: 'Google Ads', desc: 'Conversion', prefix: true, tracking: true },
    '_gcl_': { safe: false, cat: 'Google Ads', desc: 'Click-ID', prefix: true, tracking: true },
    '_dc_gtm_': { safe: false, cat: 'GTM', desc: 'Tag Manager', prefix: true, tracking: true },

    // Hotjar
    '_hj': { safe: false, cat: 'Hotjar', desc: 'Verhaltensanalyse', prefix: true, tracking: true },

    // Facebook/Meta
    '_fbp': { safe: false, cat: 'Facebook', desc: 'Pixel Tracking', tracking: true },
    '_fbc': { safe: false, cat: 'Facebook', desc: 'Click-ID', tracking: true },
    'fr': { safe: false, cat: 'Facebook', desc: 'Werbung', tracking: true },

    // LinkedIn
    'li_': { safe: false, cat: 'LinkedIn', desc: 'Tracking', prefix: true, tracking: true },
    'lidc': { safe: false, cat: 'LinkedIn', desc: 'Routing', tracking: true },
    'bcookie': { safe: false, cat: 'LinkedIn', desc: 'Browser-ID', tracking: true },

    // Twitter/X
    'guest_id': { safe: false, cat: 'Twitter', desc: 'Gast-ID', tracking: true },
    'personalization_id': { safe: false, cat: 'Twitter', desc: 'Personalisierung', tracking: true },

    // TikTok
    '_ttp': { safe: false, cat: 'TikTok', desc: 'Pixel', tracking: true },
    '_tt_': { safe: false, cat: 'TikTok', desc: 'Tracking', prefix: true, tracking: true },

    // Pinterest
    '_pinterest_': { safe: false, cat: 'Pinterest', desc: 'Tracking', prefix: true, tracking: true },
    '_pin_unauth': { safe: false, cat: 'Pinterest', desc: 'Tracking', tracking: true },

    // YouTube
    'YSC': { safe: false, cat: 'YouTube', desc: 'Session', tracking: true },
    'VISITOR_INFO1_LIVE': { safe: false, cat: 'YouTube', desc: 'Tracking', tracking: true },
    'GPS': { safe: false, cat: 'YouTube', desc: 'Standort', tracking: true },
    'PREF': { safe: false, cat: 'Google', desc: 'Einstellungen', tracking: true },

    // Vimeo
    'vuid': { safe: false, cat: 'Vimeo', desc: 'Analytics', tracking: true },

    // Jetpack
    'tk_': { safe: false, cat: 'Jetpack', desc: 'Tracking', prefix: true, tracking: true },

    // Matomo
    '_pk_': { safe: false, cat: 'Matomo', desc: 'Analytics', prefix: true, tracking: true },

    // HubSpot
    '__hs': { safe: false, cat: 'HubSpot', desc: 'Tracking', prefix: true, tracking: true },
    'hubspotutk': { safe: false, cat: 'HubSpot', desc: 'Besucher-ID', tracking: true },

    // Mailchimp
    'mailchimp': { safe: false, cat: 'Mailchimp', desc: 'Marketing', prefix: true, tracking: true },

    // Intercom
    'intercom-': { safe: false, cat: 'Intercom', desc: 'Chat-Tracking', prefix: true, tracking: true }
  }

  function getCookieInfo(name) {
    if (knownCookies[name]) return knownCookies[name]
    for (var key in knownCookies) {
      if (knownCookies[key].prefix && name.indexOf(key) === 0) {
        return knownCookies[key]
      }
    }
    return { safe: null, cat: 'Unbekannt', desc: 'Nicht in Datenbank' }
  }

  // Scanner State
  var scannerState = {
    isScanning: false,
    startTime: null,
    cookieBaseline: {},
    events: [],
    pollInterval: null
  }

  // Cookie-String zu Object parsen
  function parseCookies() {
    var cookies = {}
    var cookieStr = document.cookie
    if (!cookieStr) return cookies

    cookieStr.split(';').forEach(function (cookie) {
      var parts = cookie.trim().split('=')
      if (parts.length >= 2) {
        var name = parts[0]
        var value = parts.slice(1).join('=')
        cookies[name] = decodeURIComponent(value)
      }
    })
    return cookies
  }

  // Cookie-Wert maskieren
  function maskValue(value) {
    if (!value) return '(leer)'
    var len = value.length
    if (len <= 6) return '***' + ' (len=' + len + ')'
    var prefix = value.substring(0, 3)
    return prefix + '…' + ' (len=' + len + ')'
  }

  // Aktuellen Zeitstempel formatieren
  function formatTime(date) {
    return date.toLocaleTimeString('de-DE')
  }

  function formatDateTime(date) {
    return date.toLocaleDateString('de-DE') + ' ' + date.toLocaleTimeString('de-DE')
  }

  // Event zum Log hinzufügen
  function logEvent(action, name, value) {
    scannerState.events.push({
      action: action,
      name: name,
      valueMasked: maskValue(value),
      valueRaw: value,
      time: formatTime(new Date())
    })
    updateUI()
  }

  // Cookie-Diff prüfen
  function checkCookieDiff() {
    var currentCookies = parseCookies()
    var baseline = scannerState.cookieBaseline

    // Neue oder geänderte Cookies
    for (var name in currentCookies) {
      if (!(name in baseline)) {
        logEvent('added', name, currentCookies[name])
      } else if (baseline[name] !== currentCookies[name]) {
        logEvent('changed', name, currentCookies[name])
      }
    }

    // Entfernte Cookies
    for (var name in baseline) {
      if (!(name in currentCookies)) {
        logEvent('removed', name, '')
      }
    }

    // Baseline aktualisieren
    scannerState.cookieBaseline = currentCookies
  }

  // Scanner starten
  function startScanner() {
    if (scannerState.isScanning) return

    scannerState.isScanning = true
    scannerState.startTime = new Date()
    scannerState.events = []

    // Initial-Cookies erfassen
    var initialCookies = parseCookies()
    scannerState.cookieBaseline = initialCookies

    // Initial-Cookies als Events loggen
    for (var name in initialCookies) {
      scannerState.events.push({
        action: 'initial',
        name: name,
        valueMasked: maskValue(initialCookies[name]),
        valueRaw: initialCookies[name],
        time: formatTime(scannerState.startTime)
      })
    }

    updateUI()

    // Polling starten (alle 300ms)
    scannerState.pollInterval = setInterval(checkCookieDiff, 300)

    console.log('[Cookie-Scanner] Gestartet - ' + Object.keys(initialCookies).length + ' initiale Cookies')
  }

  // Scanner stoppen und Ergebnisse senden
  function stopScanner() {
    if (!scannerState.isScanning) return

    scannerState.isScanning = false

    if (scannerState.pollInterval) {
      clearInterval(scannerState.pollInterval)
      scannerState.pollInterval = null
    }

    // Ergebnisse an WordPress senden
    sendResults()
  }

  // Ergebnisse per AJAX senden
  function sendResults() {
    var data = new FormData()
    data.append('action', 'abw_ssd_save_cookie_scan')
    data.append('_wpnonce', abwSsdDebug.nonce)

    var scanData = {
      timestamp: formatDateTime(scannerState.startTime),
      url: window.location.href,
      events: scannerState.events
    }

    // Events als einzelne Felder senden (für PHP-Array-Parsing)
    data.append('scan_data[timestamp]', scanData.timestamp)
    data.append('scan_data[url]', scanData.url)

    scanData.events.forEach(function (event, index) {
      data.append('scan_data[events][' + index + '][action]', event.action)
      data.append('scan_data[events][' + index + '][name]', event.name)
      data.append('scan_data[events][' + index + '][valueMasked]', event.valueMasked)
      data.append('scan_data[events][' + index + '][valueRaw]', event.valueRaw)
      data.append('scan_data[events][' + index + '][time]', event.time)
    })

    fetch(abwSsdDebug.ajaxUrl, {
      method: 'POST',
      body: data,
      credentials: 'same-origin'
    })
      .then(function (response) {
        return response.json()
      })
      .then(function (result) {
        if (result.success) {
          console.log('[Cookie-Scanner] Ergebnisse gespeichert')
          showNotification('Scan-Ergebnisse gespeichert! <a href="' + abwSsdDebug.adminUrl + '" target="_blank" style="color:#fff;text-decoration:underline;">Jetzt im Admin ansehen →</a>', false, true)
        } else {
          console.error('[Cookie-Scanner] Fehler beim Speichern:', result)
          showNotification('Fehler beim Speichern: ' + (result.data || 'Unbekannt'), true)
        }
      })
      .catch(function (error) {
        console.error('[Cookie-Scanner] Netzwerkfehler:', error)
        showNotification('Netzwerkfehler beim Speichern', true)
      })
  }

  // Benachrichtigung anzeigen
  function showNotification(message, isError, isHtml) {
    var notification = document.getElementById('abw-scanner-notification')
    if (notification) {
      if (isHtml) {
        notification.innerHTML = message
      } else {
        notification.textContent = message
      }
      notification.style.background = isError ? '#dc3545' : '#28a745'
      notification.style.display = 'block'
      setTimeout(function () {
        notification.style.display = 'none'
      }, isHtml ? 8000 : 5000)
    }
  }

  // Scanner-UI erstellen
  function createUI() {
    var panel = document.createElement('div')
    panel.id = 'abw-cookie-scanner-panel'
    panel.innerHTML = [
      '<style>',
      '#abw-cookie-scanner-panel {',
      '  position: fixed;',
      '  bottom: 20px;',
      '  right: 20px;',
      '  width: 400px;',
      '  max-height: 520px;',
      '  background: #fff;',
      '  border-radius: 12px;',
      '  box-shadow: 0 8px 32px rgba(0,0,0,0.2), 0 0 0 1px rgba(139,43,209,0.2);',
      '  z-index: 999999;',
      '  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;',
      '  font-size: 13px;',
      '  overflow: hidden;',
      '}',
      '',
      '#abw-scanner-header {',
      '  background: linear-gradient(135deg, #8b2bd1 0%, #6c21a8 100%);',
      '  color: #fff;',
      '  padding: 14px 18px;',
      '  display: flex;',
      '  justify-content: space-between;',
      '  align-items: center;',
      '}',
      '#abw-scanner-header .title {',
      '  font-weight: 600;',
      '  font-size: 14px;',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 8px;',
      '}',
      '#abw-scanner-toggle {',
      '  background: rgba(255,255,255,0.2);',
      '  border: none;',
      '  color: #fff;',
      '  width: 28px;',
      '  height: 28px;',
      '  border-radius: 6px;',
      '  cursor: pointer;',
      '  font-size: 16px;',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  transition: background 0.2s;',
      '}',
      '#abw-scanner-toggle:hover { background: rgba(255,255,255,0.3); }',
      '',
      '#abw-scanner-notification {',
      '  display: none;',
      '  padding: 12px 18px;',
      '  color: #fff;',
      '  font-size: 13px;',
      '}',
      '',
      '#abw-scanner-body { padding: 16px; }',
      '',
      '#abw-scanner-status {',
      '  padding: 12px 14px;',
      '  background: #f3f4f6;',
      '  border-radius: 8px;',
      '  margin-bottom: 12px;',
      '  font-size: 13px;',
      '}',
      '#abw-scanner-status.scanning {',
      '  background: linear-gradient(135deg, #ecfdf5 0%, #d1fae5 100%);',
      '  border: 1px solid #10b981;',
      '}',
      '',
      '#abw-scanner-stats {',
      '  display: flex;',
      '  gap: 8px;',
      '  margin-top: 8px;',
      '}',
      '.abw-stat {',
      '  flex: 1;',
      '  padding: 6px 10px;',
      '  border-radius: 6px;',
      '  font-size: 12px;',
      '  font-weight: 600;',
      '  text-align: center;',
      '}',
      '.abw-stat-ok { background: #ecfdf5; color: #059669; }',
      '.abw-stat-warn { background: #fef2f2; color: #dc2626; }',
      '.abw-stat-unknown { background: #f3f4f6; color: #6b7280; }',
      '',
      '#abw-scanner-events {',
      '  max-height: 220px;',
      '  overflow-y: auto;',
      '  margin: 12px 0;',
      '  border-radius: 8px;',
      '  border: 1px solid #e5e7eb;',
      '}',
      '',
      '.abw-cookie {',
      '  display: flex;',
      '  align-items: flex-start;',
      '  gap: 12px;',
      '  padding: 12px 14px;',
      '  border-bottom: 1px solid #f3f4f6;',
      '  transition: background 0.15s;',
      '}',
      '.abw-cookie:last-child { border-bottom: none; }',
      '.abw-cookie:hover { background: #fafafa; }',
      '',
      '.abw-cookie-icon {',
      '  width: 32px;',
      '  height: 32px;',
      '  border-radius: 8px;',
      '  display: flex;',
      '  align-items: center;',
      '  justify-content: center;',
      '  font-size: 16px;',
      '  flex-shrink: 0;',
      '}',
      '.abw-cookie-ok .abw-cookie-icon { background: #ecfdf5; color: #059669; }',
      '.abw-cookie-warn .abw-cookie-icon { background: #fef2f2; color: #dc2626; }',
      '.abw-cookie-unknown .abw-cookie-icon { background: #f3f4f6; color: #9ca3af; }',
      '',
      '.abw-cookie-info { flex: 1; min-width: 0; }',
      '',
      '.abw-cookie-name {',
      '  font-family: "SF Mono", Monaco, "Consolas", monospace;',
      '  font-size: 12px;',
      '  font-weight: 600;',
      '  color: #111827;',
      '  word-break: break-all;',
      '  line-height: 1.3;',
      '}',
      '.abw-cookie-ok .abw-cookie-name { color: #059669; }',
      '.abw-cookie-warn .abw-cookie-name { color: #dc2626; }',
      '',
      '.abw-cookie-badge {',
      '  display: inline-block;',
      '  font-size: 9px;',
      '  font-weight: 600;',
      '  text-transform: uppercase;',
      '  letter-spacing: 0.5px;',
      '  padding: 2px 6px;',
      '  border-radius: 4px;',
      '  margin-left: 6px;',
      '  vertical-align: middle;',
      '}',
      '.abw-badge-new { background: #dbeafe; color: #1d4ed8; }',
      '',
      '.abw-cookie-meta {',
      '  display: flex;',
      '  align-items: center;',
      '  gap: 8px;',
      '  margin-top: 4px;',
      '  font-size: 11px;',
      '  color: #6b7280;',
      '}',
      '.abw-cookie-cat {',
      '  background: #f3f4f6;',
      '  padding: 2px 8px;',
      '  border-radius: 4px;',
      '  font-weight: 500;',
      '}',
      '.abw-cookie-ok .abw-cookie-cat { background: #ecfdf5; color: #059669; }',
      '.abw-cookie-warn .abw-cookie-cat { background: #fef2f2; color: #dc2626; }',
      '',
      '.abw-cookie-time {',
      '  color: #9ca3af;',
      '  font-size: 10px;',
      '}',
      '',
      '#abw-scanner-actions {',
      '  display: flex;',
      '  gap: 10px;',
      '  margin-top: 12px;',
      '}',
      '#abw-scanner-actions button {',
      '  flex: 1;',
      '  padding: 12px 16px;',
      '  border: none;',
      '  border-radius: 8px;',
      '  cursor: pointer;',
      '  font-weight: 600;',
      '  font-size: 13px;',
      '  transition: all 0.2s;',
      '}',
      '.abw-btn-stop {',
      '  background: #8b2bd1;',
      '  color: #fff;',
      '}',
      '.abw-btn-stop:hover { background: #6c21a8; }',
      '.abw-btn-start {',
      '  background: #059669;',
      '  color: #fff;',
      '}',
      '.abw-btn-start:hover { background: #047857; }',
      '',
      '#abw-scanner-footer {',
      '  padding: 12px 16px;',
      '  background: #f9fafb;',
      '  border-top: 1px solid #e5e7eb;',
      '  text-align: center;',
      '}',
      '#abw-btn-disable-scanner {',
      '  background: transparent;',
      '  border: 1px solid #e5e7eb;',
      '  color: #6b7280;',
      '  padding: 8px 16px;',
      '  border-radius: 6px;',
      '  cursor: pointer;',
      '  font-size: 12px;',
      '  transition: all 0.2s;',
      '}',
      '#abw-btn-disable-scanner:hover {',
      '  background: #fef2f2;',
      '  border-color: #fca5a5;',
      '  color: #dc2626;',
      '}',
      '',
      '#abw-scanner-minimized {',
      '  display: none;',
      '  padding: 16px;',
      '  text-align: center;',
      '  color: #6b7280;',
      '  cursor: pointer;',
      '}',
      '#abw-scanner-minimized:hover { background: #f9fafb; }',
      '',
      '#abw-scanner-empty {',
      '  text-align: center;',
      '  padding: 30px 20px;',
      '  color: #9ca3af;',
      '}',
      '#abw-scanner-empty svg {',
      '  width: 48px;',
      '  height: 48px;',
      '  margin-bottom: 12px;',
      '  opacity: 0.5;',
      '}',
      '</style>',
      '<div id="abw-scanner-header">',
      '  <span class="title">🍪 Cookie-Scanner</span>',
      '  <button id="abw-scanner-toggle" title="Minimieren">−</button>',
      '</div>',
      '<div id="abw-scanner-notification"></div>',
      '<div id="abw-scanner-body">',
      '  <div id="abw-scanner-status">',
      '    <div>Bereit zum Scannen</div>',
      '  </div>',
      '  <div id="abw-scanner-events">',
      '    <div id="abw-scanner-empty">',
      '      <div style="font-size:32px;margin-bottom:8px;">🔍</div>',
      '      <div>Starte den Scan, um Cookies zu erkennen</div>',
      '    </div>',
      '  </div>',
      '  <div id="abw-scanner-actions">',
      '    <button class="abw-btn-stop" id="abw-btn-stop-scan" style="display:none;">Scan beenden & speichern</button>',
      '    <button class="abw-btn-start" id="abw-btn-start-scan">Scan starten</button>',
      '  </div>',
      '</div>',
      '<div id="abw-scanner-minimized">Klicke zum Erweitern</div>',
      '<div id="abw-scanner-footer">',
      '  <button id="abw-btn-disable-scanner">Scanner deaktivieren</button>',
      '</div>'
    ].join('\n')

    document.body.appendChild(panel)

    // Event Listener
    document.getElementById('abw-btn-start-scan').addEventListener('click', function () {
      startScanner()
      this.style.display = 'none'
      document.getElementById('abw-btn-stop-scan').style.display = 'block'
    })

    document.getElementById('abw-btn-stop-scan').addEventListener('click', function () {
      stopScanner()
      this.style.display = 'none'
      document.getElementById('abw-btn-start-scan').style.display = 'block'
      document.getElementById('abw-btn-start-scan').textContent = 'Neuen Scan starten'
    })

    // Minimieren/Maximieren
    var isMinimized = false
    document.getElementById('abw-scanner-toggle').addEventListener('click', function () {
      isMinimized = !isMinimized
      document.getElementById('abw-scanner-body').style.display = isMinimized ? 'none' : 'block'
      document.getElementById('abw-scanner-minimized').style.display = isMinimized ? 'block' : 'none'
      this.textContent = isMinimized ? '+' : '−'
    })

    document.getElementById('abw-scanner-minimized').addEventListener('click', function () {
      isMinimized = false
      document.getElementById('abw-scanner-body').style.display = 'block'
      document.getElementById('abw-scanner-minimized').style.display = 'none'
      document.getElementById('abw-scanner-toggle').textContent = '−'
    })

    // Scanner deaktivieren Button
    document.getElementById('abw-btn-disable-scanner').addEventListener('click', function () {
      if (!confirm('Scanner wirklich deaktivieren?\n\nDu kannst ihn jederzeit im Admin unter "Erweitert" wieder aktivieren.')) {
        return
      }

      var btn = this
      btn.textContent = 'Wird deaktiviert...'
      btn.disabled = true

      var data = new FormData()
      data.append('action', 'abw_ssd_disable_cookie_scanner')
      data.append('_wpnonce', abwSsdDebug.nonce)

      fetch(abwSsdDebug.ajaxUrl, {
        method: 'POST',
        body: data,
        credentials: 'same-origin'
      })
        .then(function (response) {
          return response.json()
        })
        .then(function (result) {
          if (result.success) {
            showNotification('Scanner deaktiviert! Die Seite wird neu geladen...', false)
            setTimeout(function () {
              // URL ohne scan-Parameter neu laden
              var url = window.location.href.replace(/[?&]abw_cookie_scan=1/, '')
              window.location.href = url
            }, 1500)
          } else {
            btn.textContent = '✕ Scanner deaktivieren'
            btn.disabled = false
            showNotification('Fehler: ' + (result.data || 'Unbekannt'), true)
          }
        })
        .catch(function (error) {
          btn.textContent = '✕ Scanner deaktivieren'
          btn.disabled = false
          showNotification('Netzwerkfehler', true)
        })
    })

    // Auto-Start wenn URL-Parameter gesetzt
    if (window.location.search.indexOf('abw_cookie_scan=1') !== -1) {
      setTimeout(function () {
        document.getElementById('abw-btn-start-scan').click()
      }, 500)
    }
  }

  // UI aktualisieren
  function updateUI() {
    var statusEl = document.getElementById('abw-scanner-status')
    var eventsEl = document.getElementById('abw-scanner-events')

    if (!statusEl || !eventsEl) return

    // Zähle Cookie-Typen
    var safeCount = 0, trackingCount = 0, unknownCount = 0
    scannerState.events.forEach(function (e) {
      var info = getCookieInfo(e.name)
      if (info.safe === true) safeCount++
      else if (info.tracking) trackingCount++
      else unknownCount++
    })

    // Status aktualisieren
    if (scannerState.isScanning) {
      statusEl.className = 'scanning'
      statusEl.innerHTML = [
        '<div style="display:flex;align-items:center;gap:8px;">',
        '  <span style="display:inline-block;width:8px;height:8px;background:#ef4444;border-radius:50%;animation:pulse 1s infinite;"></span>',
        '  <strong>Scan läuft...</strong>',
        '</div>',
        '<div id="abw-scanner-stats">',
        '  <div class="abw-stat abw-stat-ok">✓ ' + safeCount + ' OK</div>',
        '  <div class="abw-stat abw-stat-warn">⚠ ' + trackingCount + ' Tracking</div>',
        '  <div class="abw-stat abw-stat-unknown">? ' + unknownCount + ' Unbekannt</div>',
        '</div>',
        '<style>@keyframes pulse{0%,100%{opacity:1}50%{opacity:0.5}}</style>'
      ].join('')
    } else if (scannerState.events.length > 0) {
      statusEl.className = ''
      var warning = trackingCount > 0
        ? '<div style="color:#dc2626;margin-top:8px;font-size:12px;">⚠ ' + trackingCount + ' Tracking-Cookie(s) gefunden!</div>'
        : '<div style="color:#059669;margin-top:8px;font-size:12px;">✓ Keine Tracking-Cookies erkannt</div>'
      statusEl.innerHTML = [
        '<strong>Scan beendet</strong> – ' + scannerState.events.length + ' Cookies erkannt',
        '<div id="abw-scanner-stats">',
        '  <div class="abw-stat abw-stat-ok">✓ ' + safeCount + '</div>',
        '  <div class="abw-stat abw-stat-warn">⚠ ' + trackingCount + '</div>',
        '  <div class="abw-stat abw-stat-unknown">? ' + unknownCount + '</div>',
        '</div>',
        warning
      ].join('')
    } else {
      statusEl.className = ''
      statusEl.innerHTML = '<div>Bereit zum Scannen</div>'
    }

    // Events anzeigen (Tracking zuerst, dann unbekannt, dann OK)
    var sortedEvents = scannerState.events.slice().sort(function (a, b) {
      var infoA = getCookieInfo(a.name)
      var infoB = getCookieInfo(b.name)
      var orderA = infoA.tracking ? 0 : (infoA.safe === null ? 1 : 2)
      var orderB = infoB.tracking ? 0 : (infoB.safe === null ? 1 : 2)
      if (orderA !== orderB) return orderA - orderB
      // Neue Cookies zuerst
      if (a.action === 'added' && b.action !== 'added') return -1
      if (a.action !== 'added' && b.action === 'added') return 1
      return 0
    })

    if (sortedEvents.length === 0) {
      eventsEl.innerHTML = [
        '<div id="abw-scanner-empty">',
        '  <div style="font-size:32px;margin-bottom:8px;">🔍</div>',
        '  <div>Starte den Scan, um Cookies zu erkennen</div>',
        '</div>'
      ].join('')
      return
    }

    var html = sortedEvents.map(function (event) {
      var info = getCookieInfo(event.name)
      var statusIcon = info.safe === true ? '✓' : (info.tracking ? '⚠' : '?')
      var statusClass = info.safe === true ? 'ok' : (info.tracking ? 'warn' : 'unknown')
      var isNew = event.action === 'added'
      var badgeHtml = isNew ? '<span class="abw-cookie-badge abw-badge-new">NEU</span>' : ''

      return [
        '<div class="abw-cookie abw-cookie-' + statusClass + '">',
        '  <div class="abw-cookie-icon">' + statusIcon + '</div>',
        '  <div class="abw-cookie-info">',
        '    <div class="abw-cookie-name">' + escapeHtml(event.name) + badgeHtml + '</div>',
        '    <div class="abw-cookie-meta">',
        '      <span class="abw-cookie-cat">' + escapeHtml(info.cat) + '</span>',
        '      <span class="abw-cookie-time">' + event.time + '</span>',
        '    </div>',
        '  </div>',
        '</div>'
      ].join('')
    }).join('')

    eventsEl.innerHTML = html

    // Scroll zum Anfang (Tracking-Cookies oben)
    eventsEl.scrollTop = 0
  }

  // HTML escapen
  function escapeHtml(str) {
    if (!str) return ''
    var div = document.createElement('div')
    div.textContent = str
    return div.innerHTML
  }

  // Initialisierung
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', createUI)
  } else {
    createUI()
  }

})();

